# -*- coding: utf-8 -*-

#### Importation des bibliothèques utiles

import numpy as np
import matplotlib.pyplot as plt
from scipy.integrate import odeint

#### Initialisations

"""
Résolution numérique d'une équation différentielle d'ordre 2
avec la fonction odeint de python (scipy).
v2(t) tension d'entrée du non inverseur = sortie de Wien
v1(t) tension de sortie du non inverseur = entrée de Wien
Equadiff v1"(t) + (wo/Q)v1'(t) + wo^2 v(t) = Ho(wo/Q)v2'(t)
en régime linéaire v2(t) = Gv1(t) : v1"(t) + wo / Q (1- HoG)v1'(t) + wo^2 v1(t) = 0
en regime saturé v1(t) = +/- Vsat : v1"(t) + (wo/Q)v1'(t) + wo^2 v1(t) = 0
"""

# wo pulsation propre en rad/s
fo = 1600    #Hz
wo = 2*np.pi*fo

# filtre passe-bande
Q = 1/3       # facteur de qualité
Ho = 1/3


GHo = 1.04

# Gain du non inverseur en régime linéaire
G = 3*GHo 

Vsat = 15    # V

# en régime linéaire
tau = - (2*Q)/(wo*(1 - GHo))   # constante de temps de l'exponentielle
print (" constante de temps de l'exponentielle en régime linéaire tau = {:.3E} s".format(tau))   

# Conditions initiales
v10 = 0.01                # 0.1 V tension d'offset ALI
vp0 = 0                     
y0 = [v10, vp0]           # vecteur initial

# Fonction intervenant dans l'équation différentielle vectorielle :
#     dy/dt = f(y, t) avec vecteur y = (v, vpoint)
def f(y, t) :                       # f(vecteur, temps)
    v1, vp = y
    return [vp, -wo*(1 - GHo)*vp/ Q - wo**2*v1]

# Données pour l'expérience numérique
n = 10000                    # nombre de points de l'échantillonnage
t0 = 0                      # instant initial en s
t1 = 0.015                   # instant final en s

#### Expérience numérique
t = np.linspace(t0, t1, n)
sol = odeint(f, y0, t)      # solution = liste des vecteurs pour chaque t

#### Tracés
plt.plot(t, G*sol[:, 0], label=f'v2(t) GHo = {GHo}')   #plt. instruction dans matplotlib.pyplot
plt.plot(t, sol[:, 0], label='v1(t)' )           # syntaxe plot(liste des abscisses,liste ordonnées,sol = liste des vecteurs solution
                                                # sol[:, 0] = liste des premières coordonnées des vecteurs de sol
                                                # _ : _ intervalle des valeurs de la liste = sous liste
plt.axhline(y= Vsat )
plt.axhline(y= - Vsat)
plt.legend(loc='best')                         #légende au meilleur endroit
plt.xlabel('t (s)')                             #légende axe x
plt.ylabel('v en V')                            
plt.grid()
plt.show()
 

